<?php

declare(strict_types=1);

namespace Gls\GlsPoland\Form\Type;

use Gls\GlsPoland\Configuration\Orders;
use Gls\GlsPoland\Translation\TranslatorAwareTrait;
use PrestaShop\PrestaShop\Core\Form\ChoiceProvider\OrderStateByIdChoiceProvider;
use PrestaShop\PrestaShop\Core\Form\FormChoiceProviderInterface;
use PrestaShopBundle\Form\Admin\Type\SwitchType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\ChoiceList\Loader\CallbackChoiceLoader;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

final class OrdersConfigurationType extends AbstractType
{
    use TranslatorAwareTrait;

    private $orderStatusChoices;

    /**
     * @param FormChoiceProviderInterface|OrderStateByIdChoiceProvider $orderStatusChoices
     */
    public function __construct(FormChoiceProviderInterface $orderStatusChoices)
    {
        $this->orderStatusChoices = $orderStatusChoices;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $orderChoices = new CallbackChoiceLoader([$this->orderStatusChoices, 'getChoices']);

        $builder
            ->add('labelsPrintMode', LabelsPrintModeChoiceType::class, [
                'label' => $this->getTranslator()->trans('Labels print mode', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('pickupReceiptPrintMode', PickupReceiptPrintModeChoiceType::class, [
                'label' => $this->getTranslator()->trans('Pickup receipt print mode', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('downloadLabelsAsSingleFile', SwitchType::class, [
                'required' => false,
                'label' => $this->getTranslator()->trans('Download labels as a single file', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('labelPrintedOrderStatusId', ChoiceType::class, [
                'required' => false,
                'choice_loader' => $orderChoices,
                'label' => $this->getTranslator()->trans('Order status to set after printing label', [], 'Modules.Glspoland.Configuration'),
                'placeholder' => $this->getTranslator()->trans('Do not change order status', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('consignmentDeletedOrderStatusId', ChoiceType::class, [
                'required' => false,
                'choice_loader' => $orderChoices,
                'label' => $this->getTranslator()->trans('Order status to set after deleting a consignment', [], 'Modules.Glspoland.Configuration'),
                'placeholder' => $this->getTranslator()->trans('Do not change order status', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('defaultReferences', TextType::class, [
                'required' => false,
                'label' => $this->getTranslator()->trans('Default consignment references', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('defaultNotes', TextType::class, [
                'required' => false,
                'label' => $this->getTranslator()->trans('Default consignment notes', [], 'Modules.Glspoland.Configuration'),
            ])
            ->add('defaultParcelWeightKg', NumberType::class, [
                'label' => $this->getTranslator()->trans('Default parcel weight', [], 'Modules.Glspoland.Configuration'),
                'html5' => true,
                'attr' => [
                    'step' => 'any',
                ],
                'unit' => 'kg',
                'help' => $this->getTranslator()->trans('Default weight to use if no weight is defined for the order', [], 'Modules.Glspoland.Configuration'),
            ]);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefault('data_class', Orders\InMemoryConfiguration::class);
    }
}
